/*
 
  2001-oct
  modified once again to change it back to ISA ... retaining 2.4 compatibility
  
  "Humanistic Licence Provider (HLP)"

  Device Driver for parallel port card reader interface,
  based on ISA card reader device driver of Taneem Ahmed (taneem@eyetap.org),
  which was based on ISA card reader device driver of Ben Konrath (ben@eyetap),
  based on Prof. Mann's (mann@eyetap.org) ECE 385 course project,
  http://eyetap.org/ece385/lab9, fall of 2000.
  quick hack with everything in the top half; really should be split apart
  for expandability (e.g. later data parse functions into bottom half so
  something like cat /dev/paraseat0 | letters_a-z | welcome_to_seatsale
  becomes meaningful...

  typical connections with phone wire: 
  Convention of Prof. Mann: red = +5v
                            green = clock
                            blue = data
                            black = ground 
  green sometimes goes through inverter (7404) to pin 10 of parallel port;
  In this case, one should use an inverter. 
  blue went to pin 2 of parallel port which was data 0 (D0),
  but NOW BLUE GOES TO pin13 which is slct (bit4>>4 at addr 379).
  black goes to pin 25 of parallel port (one of the grounds)
  pin9 of parallel port (d7) retracts spikes when high (e.g. outb 128)
  pin8 of parallel port (d6) activates buzzer when high (e.g. outb 64)

Looking at the parallel port at the back of the computer:
 
13  12  11  10  9   8   7   6   5   4   3   2   1
  25  24  23  22  21  20  19  18  17  16  15  14

  to get the parallel port into input mode: outb(0x30, 0x37A)
  which is hlp_base+2.
  bit 5 of base+2 is to get parallel port into bi-directional mode
                  where bidirectional mode means input mode (usually an output)
  bit 4 of base+2 is to get parallel to enable IRQ, so raising ack (pin10)
                  induces an IRQ.
  2^5+2^4 = 0x30

  ----------------------------------------------------------------------
  +5v can come from ps/2 keyboard or mouse
  diagram on beyondlogic.org or our mirror is wrong, so i drew diagram
  (looking into female connection on the computer itself) here:

     CK  NC
     _|o o|_
GND  o     o  +5V
DATA  o | o  NC
        ^
not sure on these: combination of voltmeter and comparision
against the wrong diagram on beyondlogic.org --steve (mann@eyetap.org)

ben@eyetap.org (google search), the following parallel ports are supported:
   - SIIG Cyber Parallel PCI (both versions)
   - SIIG Cyber Parallel Dual PCI (both versions)
   - SIIG Cyber 2P1S PCI
   - SIIG Cyber I/O PCI (both versions)
   - SIIG Cyber 2S1P PCI (both versions)
   - Lava Parallel PCI
   - Lava Dual Parallel PCI
   - Lava 2SP PCI
   - LavaPort Plus
this allows use on pci only system, e.g. with pci parallel port cards.

A good reference for interrupts is the chapter on interrupts in the second
edition of the Oreilly book "writing device drivers".
--yacine@eyetap.org 

*/

/*
 * Modified for ECE385 Fall 2002 Lab 7
 * Jeffery To 981781460
 */

#include <linux/module.h>
#include <linux/malloc.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/interrupt.h>

#include <linux/timer.h>
#include <asm/spinlock.h>

#include <asm/io.h>
#include <asm/uaccess.h>

#define DEFAULT_HLP_BASE	0x240
#define DEFAULT_HLP_T1_IRQ	9
#define DEFAULT_HLP_T2_IRQ	11

static short hlp_in_use;
static char hlp_name[10] = "paraseat";

static short hlp_t1_reading = 0, hlp_t2_reading = 0;
static unsigned char *t1buf, *t2buf;
static int t1read, t1count, t2read, t2count;
static struct timer_list t1timer, t2timer;
static rwlock_t t1lock = RW_LOCK_UNLOCKED, t2lock = RW_LOCK_UNLOCKED;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
  static wait_queue_head_t hlp_irq_waitQ;
#else
  static struct wait_queue* hlp_irq_waitQ;
#endif

static int hlp_major = 0;
static int hlp_t1_irq = 0;
static int hlp_t2_irq = 0;
static int hlp_base  = 0; 

MODULE_PARM(hlp_base,"i");
MODULE_PARM(hlp_t1_irq,"i");
MODULE_PARM(hlp_t2_irq,"i");

static void track_finished(unsigned long track) {
  unsigned long flags;
  if (track == 1) {
    write_lock_irqsave(&t1lock, flags);
    hlp_t1_reading = 0;
    write_unlock_irqrestore(&t1lock, flags);
  }
  else if (track == 2) {
    write_lock_irqsave(&t2lock, flags);
    hlp_t2_reading = 0;
    write_unlock_irqrestore(&t2lock, flags);
  }

  read_lock(&t1lock);
  read_lock(&t2lock);
  if (!hlp_t1_reading && !hlp_t2_reading) {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
    if(waitqueue_active(&hlp_irq_waitQ))
      wake_up(&hlp_irq_waitQ);
#else
    if(hlp_irq_waitQ)
       wake_up(&hlp_irq_waitQ);
#endif
  }
  read_unlock(&t1lock);
  read_unlock(&t2lock);
}

void hlp_handler(int irq, void *dev_id, struct pt_regs *regs) {
  if (irq == hlp_t1_irq) {
    read_lock(&t1lock);
    if (hlp_t1_reading && t1buf != NULL) {
      if (t1read < t1count) {
        t1buf[t1read] = ((inb(hlp_base)) & 0x01);
        t1read++;
        if (!timer_pending(&t1timer)) {
          t1timer.function = track_finished;
          t1timer.data = 1;
          t1timer.expires = jiffies + HZ/10;
          add_timer(&t1timer);
        }
        else {
          mod_timer(&t1timer, jiffies + HZ/10);
        }
      }
      else {
          del_timer(&t1timer);
          track_finished(1);
      }
    }
    read_unlock(&t1lock);
  }
  else if (irq == hlp_t2_irq) {
    read_lock(&t2lock);
    if (hlp_t2_reading && t2buf != NULL) {
      if (t2read < t2count) {
        t2buf[t2read] = ((inb(hlp_base)) & 0x02);
        t2read++;
        if (!timer_pending(&t2timer)) {
          t2timer.function = track_finished;
          t2timer.data = 2;
          t2timer.expires = jiffies + HZ/10;
          add_timer(&t2timer);
        }
        else {
          mod_timer(&t2timer, jiffies + HZ/10);
        }
      }
      else {
          del_timer(&t2timer);
          track_finished(2);
      }
    }
    read_unlock(&t2lock);
  }
}

static ssize_t hlp_read (
    struct file *filp,
    char *buf,        /* The buffer to fill with data */
    size_t count,     /* The length of the buffer */
    loff_t *offset){  /* Our offset in the file */

  int retval;
  unsigned char *kbuf;
  unsigned long t1flags, t2flags;
  int i;

  t1buf = (unsigned char *) kmalloc(count, GFP_KERNEL);
  t1count = count;
  t1read = 0;
  init_timer(&t1timer);

  t2buf = (unsigned char *) kmalloc(count, GFP_KERNEL);
  t2count = count;
  t2read = 0;
  init_timer(&t2timer);

  kbuf = (unsigned char *) kmalloc(count, GFP_KERNEL);

  write_lock_irqsave(&t1lock, t1flags);
  write_lock_irqsave(&t2lock, t2flags);
  hlp_t1_reading = 1;
  hlp_t2_reading = 1;
  write_unlock_irqrestore(&t1lock, t1flags);
  write_unlock_irqrestore(&t2lock, t2flags);
  interruptible_sleep_on(&hlp_irq_waitQ);

  retval = (t1read > t2read) ? t1read : t2read;
  for (i = 0; i < retval; i++) {
    kbuf[i] = 0;
    if (i < t1read)
      kbuf[i] |= (t1buf[i] & 0x01);
    if (i < t2read)
      kbuf[i] |= (t2buf[i] & 0x02);
  }

  copy_to_user(buf, kbuf, retval);
  kfree(t1buf);
  kfree(kbuf);
  return retval;
}


static int hlp_open(struct inode *inode, struct file *filp){

    if(hlp_in_use)
      return -EBUSY;
    hlp_in_use = 1;

    MOD_INC_USE_COUNT;
    return 0;
}                                     


static int hlp_release(struct inode *inode, struct file *filp){

   hlp_in_use = 0;
   MOD_DEC_USE_COUNT;
   return 0;
 }                                                       

struct file_operations hlp_fops = {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
  NULL,
#endif
  NULL,
  hlp_read,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  hlp_open,
  NULL,
  hlp_release
};

int init_module(void) {

    int result;
    unsigned long t1flags, t2flags;

    printk(KERN_INFO "Starting doing stuff\n");
    if(!hlp_t1_irq)
      hlp_t1_irq = DEFAULT_HLP_T1_IRQ;
    if(!hlp_t2_irq)
      hlp_t2_irq = DEFAULT_HLP_T2_IRQ;
    if(!hlp_base)
      hlp_base = DEFAULT_HLP_BASE;

    result = check_region(hlp_base, 8);

    if (result) {
        printk(KERN_INFO "%s: can't get I/O address 0x%x\n", hlp_name,hlp_base);
        return result;
    }
    request_region(hlp_base, 8, hlp_name); 

    result = register_chrdev(hlp_major, hlp_name, &hlp_fops);
    if (result < 0){
      printk(KERN_INFO "%s: can't get major number\n",hlp_name);
      release_region(hlp_base,8);
      return result;
    }
     
   if (!hlp_major)
     hlp_major = result;

    /*
     * request led on irq number and install handler
     */
    result = request_irq(hlp_t1_irq, hlp_handler, SA_INTERRUPT, hlp_name, NULL);
    if (result) {
      printk(KERN_INFO "%s: failed to install irq %i handler\n",hlp_name,hlp_t1_irq);
      unregister_chrdev(hlp_major, hlp_name);
      release_region(hlp_base,1);
      return result;
    }
    result = request_irq(hlp_t2_irq, hlp_handler, SA_INTERRUPT, hlp_name, NULL);
    if (result) {
      printk(KERN_INFO "%s: failed to install irq %i handler\n",hlp_name,hlp_t2_irq);
      unregister_chrdev(hlp_major, hlp_name);
      release_region(hlp_base,1);
      free_irq(hlp_t1_irq, NULL);
      return result;
    }

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,0)
    init_waitqueue_head(&hlp_irq_waitQ);
#endif

    write_lock_irqsave(&t1lock, t1flags);
    write_lock_irqsave(&t2lock, t2flags);
    hlp_t1_reading = 0;
    hlp_t2_reading = 0;
    write_unlock_irqrestore(&t1lock, t1flags);
    write_unlock_irqrestore(&t2lock, t2flags);

    printk(KERN_INFO "Done doing stuff\n");
    return 0;

}

void cleanup_module(void) {

    unregister_chrdev(hlp_major, hlp_name);
    release_region(hlp_base,8);

    free_irq(hlp_t1_irq, NULL);
    free_irq(hlp_t2_irq, NULL);
}
